# chess_fixed_orientation.py
import pygame
import chess
import chess.engine
import sys
import os
import traceback

# === تنظیمات ===
BOARD_SIZE = 600
SQUARE_SIZE = BOARD_SIZE // 8
FPS = 60

# 🔧 مسیرهای نسبی
PROJECT_ROOT = os.path.dirname(os.path.abspath(__file__))
PIECES_DIR = os.path.join(PROJECT_ROOT, "pieces")
STOCKFISH_PATH = os.path.join(PROJECT_ROOT, "..", "stockfish", "stockfish-windows-x86-64-avx2.exe")

if not os.path.isfile(STOCKFISH_PATH):
    print(f"❌ Stockfish executable not found at: {STOCKFISH_PATH}")
    sys.exit(1)

if not os.path.isdir(PIECES_DIR):
    print(f"❌ Pieces folder not found at: {PIECES_DIR}")
    sys.exit(1)

# رنگ‌ها
LIGHT_SQUARE = (240, 217, 181)
DARK_SQUARE = (181, 136, 99)
HIGHLIGHT = (186, 202, 66, 180)
SUGGESTION_HIGHLIGHT = (100, 200, 255, 150)
MOVE_HINT_COLOR = (0, 255, 0)
BACKGROUND = (30, 30, 30)
PANEL_BG = (40, 40, 40, 220)
TEXT_COLOR = (255, 255, 255)

# نگاشت مهره‌ها
PIECE_IMAGE_MAP = {
    'p': 'bp', 'r': 'br', 'n': 'bn', 'b': 'bb', 'q': 'bq', 'k': 'bk',
    'P': 'wp', 'R': 'wr', 'N': 'wn', 'B': 'wb', 'Q': 'wq', 'K': 'wk'
}

def log_error(e):
    """ثبت خطا در صورت بروز مشکل"""
    print("❌ Error occurred:")
    print(traceback.format_exc())

class ChessFixedOrientation:
    def __init__(self):
        pygame.init()
        self.screen = pygame.display.set_mode((BOARD_SIZE + 250, BOARD_SIZE))
        pygame.display.set_caption("Chess Fixed Orientation")
        self.clock = pygame.time.Clock()
        self.small_font = pygame.font.SysFont('Arial', 18)
        self.medium_font = pygame.font.SysFont('Arial', 24)
        self.large_font = pygame.font.SysFont('Arial', 32)
        self.board = chess.Board()
        self.selected_square = None
        self.valid_moves = []
        self.running = True
        self.suggested_moves = []  # لیست چند پیشنهاد اول
        self.current_suggestion_idx = 0  # ایندکس پیشنهاد فعال
        self.moving_piece_img = None
        self.moving_start_pos = None
        self.moving_end_pos = None
        self.moving_progress = 0
        self.promoting = None
        self.load_images()

    def load_images(self):
        self.images = {}
        for symbol, filename in PIECE_IMAGE_MAP.items():
            path = os.path.join(PIECES_DIR, f"{filename}.png")
            try:
                if os.path.exists(path):
                    img = pygame.image.load(path)
                    img = pygame.transform.smoothscale(img, (SQUARE_SIZE, SQUARE_SIZE))
                    self.images[symbol] = img
                else:
                    print(f"⚠️ Missing image: {path}")
            except Exception as e:
                log_error(e)

    def choose_mode(self):
        while True:
            self.screen.fill(BACKGROUND)
            title = self.large_font.render("Chess Fixed Orientation", True, TEXT_COLOR)
            text1 = self.medium_font.render("Both players play manually", True, (200, 200, 200))
            text2 = self.small_font.render("Press SPACE to start", True, (180, 180, 180))
            self.screen.blit(title, (BOARD_SIZE//2 - title.get_width()//2, 100))
            self.screen.blit(text1, (BOARD_SIZE//2 - text1.get_width()//2, 200))
            self.screen.blit(text2, (BOARD_SIZE//2 - text2.get_width()//2, 250))
            pygame.display.flip()

            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    pygame.quit()
                    sys.exit()
                elif event.type == pygame.KEYDOWN:
                    if event.key == pygame.K_SPACE:
                        return

    def draw_board(self):
        for row in range(8):
            for col in range(8):
                color = LIGHT_SQUARE if (row + col) % 2 == 0 else DARK_SQUARE
                # تغییر: الان row=0 پایین است، نه بالا
                # پس ما باید row را معکوس کنیم: 7 - row
                # اما برای نمایش، row گرافیکی همون بماند
                # اما برای تبدیل به مختصات شطرنج، باید معکوس شود
                # پس row_graphic = row
                # row_chess = 7 - row_graphic
                # پس مختصات گرافیکی: (col*SQUARE_SIZE, row*SQUARE_SIZE)
                # و مربع شطرنج: chess.square(col, 7-row)
                rect = pygame.Rect(col * SQUARE_SIZE, row * SQUARE_SIZE, SQUARE_SIZE, SQUARE_SIZE)
                pygame.draw.rect(self.screen, color, rect)

                # تبدیل مختصات گرافیکی به شطرنج
                chess_row = 7 - row  # row=0 گرافیکی می‌شود row=7 شطرنج
                chess_square = chess.square(col, chess_row)

                if self.selected_square == chess_square:
                    highlight = pygame.Surface((SQUARE_SIZE, SQUARE_SIZE), pygame.SRCALPHA)
                    highlight.fill(HIGHLIGHT)
                    self.screen.blit(highlight, rect.topleft)

                if chess_square in self.valid_moves:
                    dot = pygame.Surface((SQUARE_SIZE, SQUARE_SIZE), pygame.SRCALPHA)
                    pygame.draw.circle(dot, (0, 0, 0, 80), (SQUARE_SIZE//2, SQUARE_SIZE//2), SQUARE_SIZE//6)
                    self.screen.blit(dot, rect.topleft)

                # نمایش پیشنهاد انتخاب‌شده
                if self.suggested_moves:
                    current_move = self.suggested_moves[self.current_suggestion_idx]
                    if chess_square == current_move.from_square:
                        hsurf = pygame.Surface((SQUARE_SIZE, SQUARE_SIZE), pygame.SRCALPHA)
                        hsurf.fill(SUGGESTION_HIGHLIGHT)
                        self.screen.blit(hsurf, rect.topleft)
                    elif chess_square == current_move.to_square:
                        hsurf = pygame.Surface((SQUARE_SIZE, SQUARE_SIZE), pygame.SRCALPHA)
                        hsurf.fill(SUGGESTION_HIGHLIGHT)
                        pygame.draw.rect(hsurf, (0, 0, 255), hsurf.get_rect(), 3)
                        self.screen.blit(hsurf, rect.topleft)

    def draw_pieces(self):
        for square in chess.SQUARES:
            if self.moving_piece_img and self.moving_start_pos == square:
                continue
            piece = self.board.piece_at(square)
            if piece:
                # تبدیل مربع شطرنج به مختصات گرافیکی
                row_chess = square // 8
                col = square % 8
                row_graphic = 7 - row_chess  # row=7 شطرنج می‌شود row=0 گرافیکی
                x = col * SQUARE_SIZE
                y = row_graphic * SQUARE_SIZE
                img_key = piece.symbol()
                img = self.images.get(img_key)
                if img:
                    self.screen.blit(img, (x, y))

        if self.moving_piece_img:
            # تبدیل مربع شطرنج به مختصات گرافیکی
            start_row_chess = self.moving_start_pos // 8
            start_col = self.moving_start_pos % 8
            start_row_graphic = 7 - start_row_chess

            end_row_chess = self.moving_end_pos // 8
            end_col = self.moving_end_pos % 8
            end_row_graphic = 7 - end_row_chess

            start_x = start_col * SQUARE_SIZE
            start_y = start_row_graphic * SQUARE_SIZE
            end_x = end_col * SQUARE_SIZE
            end_y = end_row_graphic * SQUARE_SIZE

            x = start_x + (end_x - start_x) * self.moving_progress
            y = start_y + (end_y - start_y) * self.moving_progress

            self.screen.blit(self.moving_piece_img, (x, y))

    def draw_suggestions_panel(self):
        panel_x = BOARD_SIZE + 10
        panel_y = 10
        panel_width = 230
        panel_height = BOARD_SIZE - 20
        panel = pygame.Rect(panel_x, panel_y, panel_width, panel_height)
        s = pygame.Surface((panel_width, panel_height), pygame.SRCALPHA)
        s.fill(PANEL_BG)
        self.screen.blit(s, (panel_x, panel_y))

        title = self.medium_font.render("Suggestions (↑↓ to navigate):", True, TEXT_COLOR)
        self.screen.blit(title, (panel_x + 10, panel_y + 10))

        if not self.suggested_moves:
            no_move = self.small_font.render("No suggestions yet.", True, (200, 200, 200))
            self.screen.blit(no_move, (panel_x + 10, panel_y + 50))
        else:
            for i, move in enumerate(self.suggested_moves):
                color = (255, 255, 100) if i == self.current_suggestion_idx else (200, 200, 200)
                san = self.board.san(move)
                text = self.small_font.render(f"{i+1}. {san}", True, color)
                self.screen.blit(text, (panel_x + 10, panel_y + 50 + i * 30))

        note1 = self.small_font.render("You can play any legal move!", True, (150, 255, 150))
        note2 = self.small_font.render("Press R to refresh suggestions", True, (200, 200, 200))
        note3 = self.small_font.render("Turn: " + ("White" if self.board.turn == chess.WHITE else "Black"), True, (255, 255, 255))
        self.screen.blit(note1, (panel_x + 10, panel_y + panel_height - 110))
        self.screen.blit(note2, (panel_x + 10, panel_y + panel_height - 80))
        self.screen.blit(note3, (panel_x + 10, panel_y + panel_height - 50))

    def draw_move_hint_line(self):
        if self.suggested_moves:
            current_move = self.suggested_moves[self.current_suggestion_idx]
            from_sq = current_move.from_square
            to_sq = current_move.to_square

            # تبدیل مربع شطرنج به مختصات گرافیکی
            from_row_chess = from_sq // 8
            from_col = from_sq % 8
            from_row_graphic = 7 - from_row_chess

            to_row_chess = to_sq // 8
            to_col = to_sq % 8
            to_row_graphic = 7 - to_row_chess

            start_pos = (
                from_col * SQUARE_SIZE + SQUARE_SIZE//2,
                from_row_graphic * SQUARE_SIZE + SQUARE_SIZE//2
            )
            end_pos = (
                to_col * SQUARE_SIZE + SQUARE_SIZE//2,
                to_row_graphic * SQUARE_SIZE + SQUARE_SIZE//2
            )
            pygame.draw.line(self.screen, MOVE_HINT_COLOR, start_pos, end_pos, 3)
            pygame.draw.circle(self.screen, MOVE_HINT_COLOR, end_pos, 8)

    def get_square_from_mouse(self, pos):
        x, y = pos
        if x >= BOARD_SIZE:
            return None
        col = x // SQUARE_SIZE
        row_graphic = y // SQUARE_SIZE
        row_chess = 7 - row_graphic
        return chess.square(col, row_chess)

    def animate_move(self, start_sq, end_sq, piece_symbol):
        img = self.images.get(piece_symbol)
        if not img:
            self.board.push(chess.Move(start_sq, end_sq))
            return
        self.moving_piece_img = img
        self.moving_start_pos = start_sq
        self.moving_end_pos = end_sq
        self.moving_progress = 0
        while self.moving_progress < 1:
            self.moving_progress += 0.1
            self.draw_board()
            self.draw_pieces()
            self.draw_suggestions_panel()
            self.draw_move_hint_line()
            pygame.display.flip()
            self.clock.tick(FPS)
        self.moving_piece_img = None

    def get_top_moves(self, depth=5):
        try:
            with chess.engine.SimpleEngine.popen_uci(STOCKFISH_PATH) as engine:
                result = engine.analyse(self.board, chess.engine.Limit(time=1.0), multipv=depth)
                
                if isinstance(result, list):
                    moves = []
                    for entry in result:
                        if isinstance(entry, dict) and "pv" in entry and entry["pv"]:
                            moves.append(entry["pv"][0])
                    return moves[:depth]
                else:
                    if hasattr(result, 'pvs') and result.pvs:
                        return [pv[0] for pv in result.pvs if len(pv) > 0][:depth]
                    else:
                        best_result = engine.play(self.board, chess.engine.Limit(time=1.0))
                        return [best_result.move] if best_result.move else []
        except Exception as e:
            log_error(e)
            return []

    def suggest_best_moves(self):
        self.suggested_moves = self.get_top_moves(depth=5)
        self.current_suggestion_idx = 0

    def promote_dialog(self, move):
        dialog_rect = pygame.Rect(BOARD_SIZE//4, BOARD_SIZE//3, BOARD_SIZE//2, BOARD_SIZE//3)
        pygame.draw.rect(self.screen, (50, 50, 50), dialog_rect)
        pygame.draw.rect(self.screen, (200, 200, 200), dialog_rect, 2)

        options = {"Queen": chess.QUEEN, "Rook": chess.ROOK, "Bishop": chess.BISHOP, "Knight": chess.KNIGHT}
        option_names = list(options.keys())
        option_rects = []
        for i, name in enumerate(option_names):
            text = self.small_font.render(name, True, (255, 255, 255))
            x = dialog_rect.x + 20
            y = dialog_rect.y + 30 + i * 40
            self.screen.blit(text, (x, y))
            option_rects.append(pygame.Rect(x, y, text.get_width(), text.get_height()))

        pygame.display.flip()

        while True:
            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    pygame.quit()
                    sys.exit()
                elif event.type == pygame.MOUSEBUTTONDOWN:
                    for i, rect in enumerate(option_rects):
                        if rect.collidepoint(event.pos):
                            new_promo = options[option_names[i]]
                            return chess.Move(move.from_square, move.to_square, promotion=new_promo)

    def human_move(self, event):
        if event.type == pygame.MOUSEBUTTONDOWN:
            clicked_square = self.get_square_from_mouse(event.pos)
            if clicked_square is None:
                return False

            # الان کاربر می‌تونه هر دو رنگ رو بازی کنه
            piece = self.board.piece_at(clicked_square)

            if piece and piece.color == self.board.turn:
                self.selected_square = clicked_square
                self.valid_moves = [
                    move.to_square for move in self.board.legal_moves
                    if move.from_square == clicked_square
                ]
            elif self.selected_square is not None:
                move = chess.Move(self.selected_square, clicked_square)
                if self.board.piece_at(self.selected_square) and \
                   self.board.piece_at(self.selected_square).piece_type == chess.PAWN and \
                   (clicked_square // 8 == 0 or clicked_square // 8 == 7):
                    move = self.promote_dialog(move)

                if move in self.board.legal_moves:
                    piece_symbol = self.board.piece_at(move.from_square).symbol()
                    self.animate_move(move.from_square, move.to_square, piece_symbol)
                    self.board.push(move)
                    self.selected_square = None
                    self.valid_moves = []
                    self.suggest_best_moves()
                    return True
                else:
                    self.selected_square = None
                    self.valid_moves = []
        return False

    def handle_navigation_keys(self, event):
        if event.type == pygame.KEYDOWN:
            if event.key == pygame.K_UP:
                if self.suggested_moves:
                    self.current_suggestion_idx = max(0, self.current_suggestion_idx - 1)
            elif event.key == pygame.K_DOWN:
                if self.suggested_moves:
                    self.current_suggestion_idx = min(len(self.suggested_moves) - 1, self.current_suggestion_idx + 1)
            elif event.key == pygame.K_r:
                self.suggest_best_moves()

    def check_game_over(self):
        if self.board.is_game_over():
            result = self.board.result()
            if result == "1/2-1/2":
                msg = "Game Over: Draw!"
            elif result == "1-0":
                msg = "Game Over: White wins!"
            else:
                msg = "Game Over: Black wins!"
            print(msg)
            pygame.quit()
            sys.exit()

    def run(self):
        self.choose_mode()
        self.suggest_best_moves()

        while self.running:
            moved = False
            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    self.running = False
                moved = self.human_move(event)
                self.handle_navigation_keys(event)

            if moved:
                self.check_game_over()

            self.draw_board()
            self.draw_pieces()
            self.draw_suggestions_panel()
            self.draw_move_hint_line()
            pygame.display.flip()
            self.clock.tick(FPS)

        pygame.quit()

if __name__ == "__main__":
    game = ChessFixedOrientation()
    game.run()