import { useState } from "react";
import { Button } from "@/components/ui/button";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
  DropdownMenuSub,
  DropdownMenuSubContent,
  DropdownMenuSubTrigger,
} from "@/components/ui/dropdown-menu";
import { ChevronDown, Building2 } from "lucide-react";

interface DepartmentDropdownProps {
  onDepartmentSelect?: (department: string, unit?: string) => void;
}

const departments = {
  engineering: {
    name: "مهندسی",
    units: [
      "طراحی مکانیک",
      "طراحی الکترونیک", 
      "کنترل کیفیت",
      "پژوهش و توسعه",
      "مهندسی مواد",
      "مهندسی تولید"
    ]
  },
  planning: {
    name: "برنامه‌ریزی",
    units: [
      "برنامه‌ریزی تولید",
      "کنترل پروژه",
      "تدارکات",
      "زمان‌بندی",
      "منابع انسانی",
      "بودجه‌بندی"
    ]
  },
  commercial: {
    name: "بازرگانی",
    units: [
      "فروش داخلی",
      "صادرات",
      "بازاریابی",
      "خدمات مشتریان",
      "قراردادها",
      "توزیع"
    ]
  },
  quality: {
    name: "تضمین کیفیت",
    units: [
      "کنترل کیفیت",
      "تضمین کیفیت",
      "آزمون‌های مواد",
      "استانداردسازی",
      "مستندسازی",
      "بازرسی"
    ]
  },
  financial: {
    name: "مالی",
    units: [
      "حسابداری",
      "خزانه‌داری",
      "بودجه",
      "حسابرسی داخلی",
      "مالیات",
      "گزارشات مالی"
    ]
  },
  laboratory: {
    name: "آزمایشگاه",
    units: [
      "آزمون‌های شیمیایی",
      "آزمون‌های فیزیکی",
      "میکروبیولوژی",
      "کالیبراسیون",
      "آنالیز مواد",
      "تست محیطی"
    ]
  },
  warehouse: {
    name: "انبار",
    units: [
      "انبار مواد اولیه",
      "انبار محصولات",
      "لجستیک",
      "حمل و نقل",
      "بسته‌بندی",
      "موجودی"
    ]
  }
};

export default function DepartmentDropdown({ onDepartmentSelect }: DepartmentDropdownProps) {
  const [selectedDepartment, setSelectedDepartment] = useState<string>("");
  const [selectedUnit, setSelectedUnit] = useState<string>("");

  const handleDepartmentClick = (deptKey: string, deptName: string) => {
    setSelectedDepartment(deptName);
    setSelectedUnit("");
    onDepartmentSelect?.(deptKey);
  };

  const handleUnitClick = (deptKey: string, unitName: string) => {
    setSelectedUnit(unitName);
    onDepartmentSelect?.(deptKey, unitName);
  };

  return (
    <div className="flex items-center gap-2">
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="outline" className="min-w-[200px] justify-between">
            <div className="flex items-center gap-2">
              <Building2 className="h-4 w-4" />
              <span>
                {selectedDepartment 
                  ? `${selectedDepartment}${selectedUnit ? ` - ${selectedUnit}` : ''}`
                  : "انتخاب واحد"
                }
              </span>
            </div>
            <ChevronDown className="h-4 w-4" />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent className="w-56" align="end">
          {Object.entries(departments).map(([deptKey, dept]) => (
            <DropdownMenuSub key={deptKey}>
              <DropdownMenuSubTrigger 
                onClick={() => handleDepartmentClick(deptKey, dept.name)}
                className="cursor-pointer"
              >
                <Building2 className="mr-2 h-4 w-4" />
                <span>{dept.name}</span>
              </DropdownMenuSubTrigger>
              <DropdownMenuSubContent>
                {dept.units.map((unit) => (
                  <DropdownMenuItem
                    key={unit}
                    onClick={() => handleUnitClick(deptKey, unit)}
                    className="cursor-pointer"
                  >
                    {unit}
                  </DropdownMenuItem>
                ))}
              </DropdownMenuSubContent>
            </DropdownMenuSub>
          ))}
        </DropdownMenuContent>
      </DropdownMenu>
    </div>
  );
}