import { useState } from "react";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Building, User, Lock } from "lucide-react";
import supplyChainBg2 from "@/assets/supply-chain-bg2.jpg";

export default function Landing() {
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const { toast } = useToast();

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!username || !password) {
      toast({
        title: "خطا",
        description: "نام کاربری و رمز عبور الزامی است",
        variant: "destructive",
      });
      return;
    }

    setIsLoading(true);
    
    try {
      const response = await apiRequest("POST", "/api/auth/login", {
        username,
        password,
      });

      const data = await response.json();
      
      toast({
        title: "موفقیت",
        description: "ورود با موفقیت انجام شد",
      });

      // Reload to trigger auth state update
      window.location.reload();
    } catch (error: any) {
      toast({
        title: "خطا در ورود",
        description: error.message || "نام کاربری یا رمز عبور اشتباه است",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex flex-row-reverse">
      {/* Right section - 70% of screen with background image */}
      <div 
        className="hidden lg:block relative"
        style={{
          width: '70%',
          backgroundImage: `url(${supplyChainBg2})`,
          backgroundSize: 'cover',
          backgroundPosition: 'center',
          backgroundRepeat: 'no-repeat'
        }}
      >
        {/* Overlay for better visual effect */}
        <div className="absolute inset-0 bg-gradient-to-l from-blue-900/20 to-cyan-900/20"></div>
      </div>

      {/* Left section - 30% of screen with login form */}
      <div 
        className="bg-white flex items-center justify-center p-8"
        style={{ width: '30%', minWidth: '400px' }}
      >
        <div className="w-full max-w-md">
          <div className="text-center mb-8">
            <div className="w-20 h-20 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-6 shadow-lg">
              <Building className="text-white text-2xl" />
            </div>
            <h1 className="text-2xl font-bold text-gray-800 mb-2">
              سیستم تامین نمونه کالای جدید
            </h1>
            <p className="text-lg text-gray-600 mb-1">
              مدیریت زنجیره تامین هوشمند
            </p>
            <p className="text-sm text-gray-500 mb-6">شرکت سامفر</p>
            <h2 className="text-xl font-semibold text-gray-700 mb-4">
              ورود به سیستم
            </h2>
          </div>
          
          <form onSubmit={handleLogin} className="space-y-6">
            <div className="space-y-2">
              <Label htmlFor="username" className="text-sm font-medium text-gray-700">
                نام کاربری
              </Label>
              <div className="relative">
                <User className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                <Input
                  id="username"
                  type="text"
                  value={username}
                  onChange={(e) => setUsername(e.target.value)}
                  className="pr-10 border-gray-300 focus:border-blue-500 focus:ring-blue-500"
                  placeholder="نام کاربری خود را وارد کنید"
                  disabled={isLoading}
                />
              </div>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="password" className="text-sm font-medium text-gray-700">
                رمز عبور
              </Label>
              <div className="relative">
                <Lock className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                <Input
                  id="password"
                  type="password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  className="pr-10 border-gray-300 focus:border-blue-500 focus:ring-blue-500"
                  placeholder="رمز عبور خود را وارد کنید"
                  disabled={isLoading}
                />
              </div>
            </div>
            
            <Button 
              type="submit" 
              className="w-full bg-blue-500 hover:bg-blue-600 text-white font-medium py-3 transition-colors" 
              disabled={isLoading}
            >
              {isLoading ? "در حال ورود..." : "ورود به سیستم"}
            </Button>
          </form>
        </div>
      </div>

      {/* Mobile layout for small screens */}
      <div className="lg:hidden w-full bg-white flex items-center justify-center p-8">
        <div className="w-full max-w-md">
          <div className="text-center mb-8">
            <div className="w-20 h-20 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-6 shadow-lg">
              <Building className="text-white text-2xl" />
            </div>
            <h1 className="text-xl font-bold text-gray-800 mb-2">
              سیستم تامین نمونه کالای جدید
            </h1>
            <p className="text-md text-gray-600 mb-1">
              مدیریت زنجیره تامین هوشمند
            </p>
            <p className="text-sm text-gray-500 mb-6">شرکت سامفر</p>
            <h2 className="text-lg font-semibold text-gray-700 mb-4">
              ورود به سیستم
            </h2>
          </div>
          
          <form onSubmit={handleLogin} className="space-y-6">
            <div className="space-y-2">
              <Label htmlFor="username-mobile" className="text-sm font-medium text-gray-700">
                نام کاربری
              </Label>
              <div className="relative">
                <User className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                <Input
                  id="username-mobile"
                  type="text"
                  value={username}
                  onChange={(e) => setUsername(e.target.value)}
                  className="pr-10 border-gray-300 focus:border-blue-500 focus:ring-blue-500"
                  placeholder="نام کاربری خود را وارد کنید"
                  disabled={isLoading}
                />
              </div>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="password-mobile" className="text-sm font-medium text-gray-700">
                رمز عبور
              </Label>
              <div className="relative">
                <Lock className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                <Input
                  id="password-mobile"
                  type="password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  className="pr-10 border-gray-300 focus:border-blue-500 focus:ring-blue-500"
                  placeholder="رمز عبور خود را وارد کنید"
                  disabled={isLoading}
                />
              </div>
            </div>
            
            <Button 
              type="submit" 
              className="w-full bg-blue-500 hover:bg-blue-600 text-white font-medium py-3 transition-colors" 
              disabled={isLoading}
            >
              {isLoading ? "در حال ورود..." : "ورود به سیستم"}
            </Button>
          </form>
        </div>
      </div>
    </div>
  );
}
