import { sql } from 'drizzle-orm';
import {
  index,
  jsonb,
  pgTable,
  timestamp,
  varchar,
  text,
  integer,
  boolean,
} from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Session storage table (mandatory for Replit Auth)
export const sessions = pgTable(
  "sessions",
  {
    sid: varchar("sid").primaryKey(),
    sess: jsonb("sess").notNull(),
    expire: timestamp("expire").notNull(),
  },
  (table) => [index("IDX_session_expire").on(table.expire)],
);

// User storage table (mandatory for Replit Auth)
export const users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  email: varchar("email").unique(),
  firstName: varchar("first_name"),
  lastName: varchar("last_name"),
  profileImageUrl: varchar("profile_image_url"),
  signatureImageUrl: varchar("signature_image_url"), // Digital signature image
  username: varchar("username").unique().notNull(),
  password: varchar("password").notNull(),
  department: varchar("department").notNull(), // engineering, planning, commercial, quality, financial, laboratory, warehouse, admin
  position: varchar("position"), // مدیر / سرپرست
  fullName: varchar("full_name").notNull(),
  departmentAccess: jsonb("department_access").default([]), // Array of departments user has access to
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Forms table
export const forms = pgTable("forms", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  formNumber: varchar("form_number").unique().notNull(),
  currentDepartment: varchar("current_department").notNull(),
  assignedToUserId: varchar("assigned_to_user_id").references(() => users.id), // For delegation
  delegatedByUserId: varchar("delegated_by_user_id").references(() => users.id), // Who delegated this form
  status: varchar("status").notNull().default('pending'), // pending, in_progress, completed, rejected
  createdBy: varchar("created_by").notNull().references(() => users.id),
  creatorName: varchar("creator_name"), // Name of the form creator
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
  archivedAt: timestamp("archived_at"), // When form was archived
  
  // Engineering section - specific fields from the form
  suggestedSupplier: varchar("suggested_supplier"), // تامین کننده پیشنهادی
  drawingCode: varchar("drawing_code"), // کد نقشه
  sentDocuments: varchar("sent_documents"), // مدارک ارسالی
  documentSendDate: varchar("document_send_date"), // تاریخ ارسال مدارک (Persian date)
  needDate: varchar("need_date"), // تاریخ نیاز (Persian date)
  productName: varchar("product_name"), // نام محصول
  supplyExecutive: varchar("supply_executive"), // مجری تامین
  supplyOrder: varchar("supply_order"), // مرتبه تامین
  newProductDevPhase: varchar("new_product_dev_phase"), // فاز تکوین محصول جدید
  issueReason: varchar("issue_reason"), // علت صدور
  indicatorNumber: varchar("indicator_number"), // شماره اندیکاتور
  actionNumber: varchar("action_number"), // شماره اقدام
  engineeringDate: varchar("engineering_date"), // تاریخ (Persian date)
  previousFormIndicator: varchar("previous_form_indicator"), // شماره اندیکاتور فرم قبلی
  projectCostCenterCode: varchar("project_cost_center_code"), // کد مرکز هزینه پروژه
  
  // Parts table (3 columns: نام قطعه، مقدار درخواستی، واحد)
  partsTable: jsonb("parts_table").default([]),
  
  // Approval signatures
  expertApproval: varchar("expert_approval"), // تایید کارشناس/سرپرست
  expertApprovalImageUrl: varchar("expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  managerApproval: varchar("manager_approval"), // مدیر/جانشین
  managerApprovalImageUrl: varchar("manager_approval_image_url"), // تصویر امضای مدیر/جانشین
  
  engineeringCompleted: boolean("engineering_completed").default(false),
  engineeringCompletedBy: varchar("engineering_completed_by").references(() => users.id),
  engineeringCompletedAt: timestamp("engineering_completed_at"),
  
  // Planning section - specific fields from the form
  planningExecutionDate: varchar("planning_execution_date"), // تاریخ انجام (Persian date)
  planningMonitoringTable: jsonb("planning_monitoring_table").default([]), // ثبت در فایل پایش
  planningExpertApproval: varchar("planning_expert_approval"), // تایید کارشناس/سرپرست
  planningExpertApprovalImageUrl: varchar("planning_expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  planningManagerApproval: varchar("planning_manager_approval"), // تایید مدیر (legacy field)
  planningManagerApprovalImageUrl: varchar("planning_manager_approval_image_url"), // تصویر امضای مدیر (legacy field)
  
  planningCompleted: boolean("planning_completed").default(false),
  planningCompletedBy: varchar("planning_completed_by").references(() => users.id),
  planningCompletedAt: timestamp("planning_completed_at"),
  
  // Commercial section - specific fields from the form
  commercialSupplierNameCode: varchar("commercial_supplier_name_code"), // نام/کد تامین کننده
  commercialSampleSendDate: varchar("commercial_sample_send_date"), // تاریخ ارسال نمونه (Persian date)
  commercialBillNumber: varchar("commercial_bill_number"), // شماره بارنامه
  commercialShipmentCode: varchar("commercial_shipment_code"), // کد محموله
  commercialContractorCertificate: varchar("commercial_contractor_certificate"), // گواهی پیمانکار
  commercialReceivedTable: jsonb("commercial_received_table").default([]), // جدول مقدار دریافتی
  commercialExpertApproval: varchar("commercial_expert_approval"), // تایید کارشناس/سرپرست
  commercialExpertApprovalImageUrl: varchar("commercial_expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  commercialManagerApproval: varchar("commercial_manager_approval"), // تایید مدیر/جانشین
  commercialManagerApprovalImageUrl: varchar("commercial_manager_approval_image_url"), // تصویر امضای مدیر/جانشین
  
  commercialCompleted: boolean("commercial_completed").default(false),
  commercialCompletedBy: varchar("commercial_completed_by").references(() => users.id),
  commercialCompletedAt: timestamp("commercial_completed_at"),
  
  // Quality Assurance section - specific fields from the form
  qualityExecutionDate: varchar("quality_execution_date"), // تاریخ انجام (Persian date)
  qualityPartsTable: jsonb("quality_parts_table").default([]), // جدول قطعات (کد قطعه، واحد، نوع کالای مصرفی)
  qualityExpertApproval: varchar("quality_expert_approval"), // تایید کارشناس/سرپرست
  qualityExpertApprovalImageUrl: varchar("quality_expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  qualityManagerApproval: varchar("quality_manager_approval"), // تایید مدیر/جانشین
  qualityManagerApprovalImageUrl: varchar("quality_manager_approval_image_url"), // تصویر امضای مدیر/جانشین
  
  qualityCompleted: boolean("quality_completed").default(false),
  qualityCompletedBy: varchar("quality_completed_by").references(() => users.id),
  qualityCompletedAt: timestamp("quality_completed_at"),
  
  // Financial section - specific fields from the form
  financialExecutionDate: varchar("financial_execution_date"), // تاریخ انجام (Persian date)
  financialCodeTable: jsonb("financial_code_table").default([]), // جدول ثبت کد
  financialExpertApproval: varchar("financial_expert_approval"), // تایید کارشناس/سرپرست
  financialExpertApprovalImageUrl: varchar("financial_expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  
  financialCompleted: boolean("financial_completed").default(false),
  financialCompletedBy: varchar("financial_completed_by").references(() => users.id),
  financialCompletedAt: timestamp("financial_completed_at"),
  
  // Laboratory Testing section - specific fields from the form
  laboratorySpecialStorageConditions: varchar("laboratory_special_storage_conditions"), // شرایط خاص نگهداری
  laboratoryExecutionDate: varchar("laboratory_execution_date"), // تاریخ انجام (Persian date)
  laboratoryFinalStatus: varchar("laboratory_final_status"), // وضعیت نهایی قطعه (OK, N.OK, مشروط)
  laboratoryTestRequestNumber: varchar("laboratory_test_request_number"), // شماره درخواست آزمایش
  laboratoryTestRequestDate: varchar("laboratory_test_request_date"), // تاریخ درخواست
  laboratoryPartsStatusTable: jsonb("laboratory_parts_status_table").default([]), // جدول وضعیت قطعات (نام قطعه، وضعیت نهایی)
  laboratoryTestsTable: jsonb("laboratory_tests_table").default([]), // جدول آزمون‌ها (نام آزمون، روش آزمون، نتیجه آزمون، وضعیت قطعه، ملاحظات)
  laboratoryExpertApproval: varchar("laboratory_expert_approval"), // تایید کارشناس/سرپرست
  laboratoryExpertApprovalImageUrl: varchar("laboratory_expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  laboratoryManagerApproval: varchar("laboratory_manager_approval"), // تایید مدیر/جانشین
  laboratoryManagerApprovalImageUrl: varchar("laboratory_manager_approval_image_url"), // تصویر امضای مدیر/جانشین
  
  // Dimensional Measurement Recording section
  dimensionalPartName: varchar("dimensional_part_name"), // نام قطعه
  dimensionalLotCode: varchar("dimensional_lot_code"), // کد محموله
  dimensionalDrawingTestNumber: varchar("dimensional_drawing_test_number"), // شماره نقشه/طرح تست
  dimensionalSupplier: varchar("dimensional_supplier"), // تامین کننده
  dimensionalSampleSupplyNumber: varchar("dimensional_sample_supply_number"), // شماره تامین نمونه کالا
  dimensionalSpecialStorageConditions: varchar("dimensional_special_storage_conditions"), // شرایط خاص نگهداری
  dimensionalMeasurementsTable: jsonb("dimensional_measurements_table").default([]), // جدول ثبت مقادیر اندازه گیری ابعادی (پارامتر کنترلی، حدود پذیرش، شماره نمونه، توضیحات)
  dimensionalLotStatus: varchar("dimensional_lot_status"), // وضعیت محموله (تایید، مردود)
  
  laboratoryCompleted: boolean("laboratory_completed").default(false),
  laboratoryCompletedBy: varchar("laboratory_completed_by").references(() => users.id),
  laboratoryCompletedAt: timestamp("laboratory_completed_at"),
  
  // Warehouse section - specific fields from the form
  warehouseDeliveryDate: varchar("warehouse_delivery_date"), // تاریخ تحویل نمونه‌ها (Persian date)
  warehouseHealthyCount: varchar("warehouse_healthy_count"), // تعداد سالم
  warehouseHealthyUnit: varchar("warehouse_healthy_unit"), // واحد سالم
  warehouseWasteCount: varchar("warehouse_waste_count"), // تعداد ضایعات
  warehouseWasteUnit: varchar("warehouse_waste_unit"), // واحد ضایعات
  warehouseExpertApproval: varchar("warehouse_expert_approval"), // تایید کارشناس/سرپرست
  warehouseExpertApprovalImageUrl: varchar("warehouse_expert_approval_image_url"), // تصویر امضای کارشناس/سرپرست
  
  warehouseCompleted: boolean("warehouse_completed").default(false),
  warehouseCompletedBy: varchar("warehouse_completed_by").references(() => users.id),
  warehouseCompletedAt: timestamp("warehouse_completed_at"),
  
  // Archive functionality
  isArchived: boolean("is_archived").default(false),
});

// Form history for audit trail
export const formHistory = pgTable("form_history", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  formId: varchar("form_id").notNull().references(() => forms.id),
  fromDepartment: varchar("from_department"),
  toDepartment: varchar("to_department").notNull(),
  action: varchar("action").notNull(), // created, forwarded, completed, rejected, returned
  userId: varchar("user_id").notNull().references(() => users.id),
  assignedToUserId: varchar("assigned_to_user_id").references(() => users.id), // Who form is assigned to
  notes: text("notes"),
  isReturned: boolean("is_returned").default(false), // Track returns/revisions
  signatureApplied: boolean("signature_applied").default(false), // Track digital signatures
  signatureData: text("signature_data"), // JSON signature data
  receivedAt: timestamp("received_at").defaultNow(), // When form was received by department
  processedAt: timestamp("processed_at"), // When form was signed/processed
  forwardedAt: timestamp("forwarded_at"), // When form was forwarded to next department
  processingTimeMinutes: integer("processing_time_minutes"), // Time taken to process
  createdAt: timestamp("created_at").defaultNow(),
});

// Form assignments table for tracking current assignments
export const formAssignments = pgTable("form_assignments", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  formId: varchar("form_id").notNull().references(() => forms.id),
  currentDepartment: varchar("current_department").notNull(),
  assignedToUserId: varchar("assigned_to_user_id").notNull().references(() => users.id),
  assignedBy: varchar("assigned_by").notNull().references(() => users.id),
  assignedAt: timestamp("assigned_at").defaultNow(),
  isActive: boolean("is_active").default(true),
  notes: text("notes"),
});

// Insert schemas
export const insertUserSchema = createInsertSchema(users).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  signatureImageUrl: z.string().optional(),
});

export const insertFormSchema = createInsertSchema(forms).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertFormHistorySchema = createInsertSchema(formHistory).omit({
  id: true,
  createdAt: true,
});

export const insertFormAssignmentSchema = createInsertSchema(formAssignments).omit({
  id: true,
  assignedAt: true,
});

// Types
export type UpsertUser = typeof users.$inferInsert;
export type User = typeof users.$inferSelect;
export type InsertUser = z.infer<typeof insertUserSchema>;
export type Form = typeof forms.$inferSelect;
export type InsertForm = z.infer<typeof insertFormSchema>;
export type FormHistory = typeof formHistory.$inferSelect;
export type InsertFormHistory = z.infer<typeof insertFormHistorySchema>;
export type FormAssignment = typeof formAssignments.$inferSelect;
export type InsertFormAssignment = z.infer<typeof insertFormAssignmentSchema>;

// Department enum
export const DEPARTMENTS = {
  ENGINEERING: 'engineering',
  PLANNING: 'planning',
  COMMERCIAL: 'commercial',
  QUALITY: 'quality',
  FINANCIAL: 'financial',
  LABORATORY: 'laboratory',
  WAREHOUSE: 'warehouse',
  ADMIN: 'admin'
} as const;

export const DEPARTMENT_NAMES = {
  [DEPARTMENTS.ENGINEERING]: 'مهندسی',
  [DEPARTMENTS.PLANNING]: 'برنامه‌ریزی',
  [DEPARTMENTS.COMMERCIAL]: 'بازرگانی',
  [DEPARTMENTS.QUALITY]: 'تضمین کیفیت',
  [DEPARTMENTS.FINANCIAL]: 'مالی',
  [DEPARTMENTS.LABORATORY]: 'تست آزمایشگاهی',
  [DEPARTMENTS.WAREHOUSE]: 'انبار',
  [DEPARTMENTS.ADMIN]: 'مدیریت'
} as const;

export const DEPARTMENT_ORDER = [
  DEPARTMENTS.ENGINEERING,
  DEPARTMENTS.PLANNING,
  DEPARTMENTS.COMMERCIAL,
  DEPARTMENTS.QUALITY,
  DEPARTMENTS.FINANCIAL,
  DEPARTMENTS.LABORATORY,
  DEPARTMENTS.WAREHOUSE
] as const;

// Form-specific schemas
export const planningFormSchema = createInsertSchema(forms).pick({
  planningExecutionDate: true,
  planningExpertApproval: true,
  planningExpertApprovalImageUrl: true,
  planningManagerApproval: true,
});

export type PlanningFormData = z.infer<typeof planningFormSchema>;
